// ----------------------------------------------------------------------------
// hwfall_viewer - horizontal waterfall / spectrum view for long duration modes
//
// Copyright (C) 2006-2010
//		Dave Freese, W1HKJ
//		   (C) Mauri Niininen, AG1LE
//
// This file is part of fldigi.  Adapted from code contained in gmfsk source code
// distribution.
//  gmfsk Copyright (C) 2001, 2002, 2003
//  Tomi Manninen (oh2bns@sral.fi)
//  Copyright (C) 2004
//  Lawrence Glaister (ve7it@shaw.ca)
//
// Fldigi is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Fldigi is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with fldigi.  If not, see <http://www.gnu.org/licenses/>.
// ----------------------------------------------------------------------------

#include <iostream>
#include "configuration.h"
#include "hwfall_viewer.h"
#include "hwfall.h"
#include "plot_xy.h"
#include "util.h"

#include "modem.h"
#include "filters.h"
#include "fftfilt.h"
#include "mbuffer.h"
#include "fl_digi.h"
#include "threads.h"
#include "trx.h"

#define NUM_BINS 500 // change in units of 100

sfft		*hwfall_viewer_sfft;
cmplx		sfft_bins[NUM_BINS];

Cmovavg		*hwfall_viewer_movavg[NUM_BINS];
double		bins[NUM_BINS];

int			hwfall_start_freq = 200;

plot_xy*    hwfall_fft_plot = (plot_xy *)0;
hwfall*     hwfall_wfall = (hwfall *)0;

Fl_Counter* hwfall_viewer_range = (Fl_Counter *)0;
Fl_Counter* hwfall_viewer_min_db = (Fl_Counter *)0;
Fl_Counter* hwfall_viewer_start = (Fl_Counter *)0;
Fl_Counter* hwfall_waterfall_speed = (Fl_Counter *)0;

Fl_Choice*  hwfall_viewer_palette = (Fl_Choice *)0;
Fl_Choice*	hwfall_lowpass = (Fl_Choice *)0;

Fl_Button*  btn_hwfall_clear_display = (Fl_Button *)0;

Fl_Light_Button* btn_pause_display = (Fl_Light_Button *)0;

static void cb_hwfall_viewer_min_db(Fl_Counter* o, void*) {
  progdefaults.hwfall_viewer_min_db = o->value();
  progdefaults.changed = true;
}

static void cb_hwfall_viewer_range(Fl_Counter* o, void*) {
  progdefaults.hwfall_viewer_range = o->value();
  progdefaults.changed = true;
}

static void cb_hwfall_viewer_palette (Fl_Choice* o, void*) {
	if (o->value() >= 0 && o->value() < 27) {
		progdefaults.hwfall_palette = o->value();
		hwfall_wfall->setcolors(progdefaults.hwfall_palette);
		progdefaults.changed = true;
	}
}

static void cb_hwfall_lowpass (Fl_Choice* o, void*) {
	if (o->value() >= 0 && o->value() < 27) {
		progdefaults.hwfall_average_selection = o->value();
		progdefaults.changed = true;
	}
}

static double disp_time = 0;
static double disp_next = 1.0;

static void cb_hwfall_waterfall_speed (Fl_Counter* o, void*) {
	progdefaults.hwfall_waterfall_rate = o->value();
	disp_time = disp_next = 0;
	progdefaults.changed = true;
}

static void cb_hwfall_clear_display (Fl_Choice* o, void*) {
	hwfall_wfall->clear();
	disp_time = 0.0;
	disp_next = 0.5;
}

static void cb_hwfall_viewer_start (Fl_Counter* o, void *) {
	hwfall_init();
}

int average_over[] = {0, 16, 8, 4};
int average_selection = 1;

Fl_Double_Window* create_hwfall_viewer() {

// signal processing
	hwfall_viewer_sfft = new sfft(active_modem->get_samplerate(), hwfall_start_freq, hwfall_start_freq + NUM_BINS);

	for (size_t n = 0; n < NUM_BINS; n++) hwfall_viewer_movavg[n] = new Cmovavg(average_over[average_selection]);

	Fl_Double_Window* w = new Fl_Double_Window(670, NUM_BINS + 60, "Waterfall Signal Viewer");

		hwfall_fft_plot = new plot_xy(5, 5, 60, NUM_BINS , "");
		hwfall_fft_plot->box(FL_DOWN_BOX);
		hwfall_fft_plot->x_scale(0, 100, 5);
		hwfall_fft_plot->y_scale(0, NUM_BINS, NUM_BINS / 100);
		hwfall_fft_plot->bk_color(FL_BLACK);
		hwfall_fft_plot->axis_color(fl_rgb_color(192,192,192));
		hwfall_fft_plot->line_color_1(fl_rgb_color(192,192,192));
		hwfall_fft_plot->plot_over_axis(true);
		hwfall_fft_plot->legends(false);
		hwfall_fft_plot->set_borders(false);
		hwfall_fft_plot->reverse_x(true);

		hwfall_wfall = new hwfall(
			hwfall_fft_plot->x() + hwfall_fft_plot->w(), hwfall_fft_plot->y(),
			600, NUM_BINS);
		hwfall_wfall->box(FL_THIN_DOWN_BOX);
		hwfall_wfall->color(FL_BLACK);
		hwfall_wfall->selection_color(FL_BACKGROUND_COLOR);
		hwfall_wfall->labeltype(FL_NORMAL_LABEL);
		hwfall_wfall->labelfont(0);
		hwfall_wfall->labelsize(14);
		hwfall_wfall->labelcolor(FL_FOREGROUND_COLOR);
		hwfall_wfall->setcolors(progdefaults.hwfall_palette);
		hwfall_wfall->align(Fl_Align(FL_ALIGN_CENTER));
		hwfall_wfall->when(FL_WHEN_RELEASE);
		hwfall_wfall->set_miny(0);
		hwfall_wfall->tooltip("Left click for frequeny");

//----------------------------------------------------------------------

		hwfall_viewer_min_db = new Fl_Counter( 
			hwfall_fft_plot->x(), hwfall_fft_plot->y() + hwfall_fft_plot->h() + 10,
			90, 22, "Minimum (db)");
		hwfall_viewer_min_db->value(progdefaults.hwfall_viewer_min_db);
		hwfall_viewer_min_db->step(1);
		hwfall_viewer_min_db->lstep(10);
		hwfall_viewer_min_db->minimum(-60);
		hwfall_viewer_min_db->maximum(60);
		hwfall_viewer_min_db->align(Fl_Align(FL_ALIGN_BOTTOM));
		hwfall_viewer_min_db->callback((Fl_Callback*)cb_hwfall_viewer_min_db);

		hwfall_viewer_range = new Fl_Counter( 
			hwfall_viewer_min_db->x() + hwfall_viewer_min_db->w() + 5, hwfall_viewer_min_db->y(),
			90, 22, "Range (db)");
		hwfall_viewer_range->value(progdefaults.hwfall_viewer_range);
		hwfall_viewer_range->step(1);
		hwfall_viewer_range->lstep(10);
		hwfall_viewer_range->minimum(20);
		hwfall_viewer_range->maximum(140);
		hwfall_viewer_range->align(Fl_Align(FL_ALIGN_BOTTOM));
		hwfall_viewer_range->callback((Fl_Callback*)cb_hwfall_viewer_range);

		hwfall_viewer_start = new Fl_Counter(
			hwfall_viewer_range->x() + hwfall_viewer_range->w() + 5, hwfall_viewer_min_db->y(),
			90, 22, "Start Freq");
		hwfall_viewer_start->value(200);
		hwfall_viewer_start->step(100);
		hwfall_viewer_start->lstep(500);
		hwfall_viewer_start->minimum(0);
		hwfall_viewer_start->maximum(3400);
		hwfall_viewer_start->align(Fl_Align(FL_ALIGN_BOTTOM));
		hwfall_viewer_start->callback((Fl_Callback*)cb_hwfall_viewer_start);

		hwfall_waterfall_speed = new Fl_Counter(
			hwfall_viewer_start->x() + hwfall_viewer_start->w() + 5, hwfall_viewer_min_db->y(),
			90, 22, "Update");
		hwfall_waterfall_speed->value(1.0);
		hwfall_waterfall_speed->step(0.5);
		hwfall_waterfall_speed->lstep(5.0);
		hwfall_waterfall_speed->minimum(0.5);
		hwfall_waterfall_speed->maximum(30.0);
		hwfall_waterfall_speed->align(Fl_Align(FL_ALIGN_BOTTOM));
		hwfall_waterfall_speed->callback((Fl_Callback*)cb_hwfall_waterfall_speed);
		hwfall_waterfall_speed->tooltip("Update rate in seconds");

		hwfall_lowpass = new Fl_Choice(
			hwfall_waterfall_speed->x() + hwfall_waterfall_speed->w() + 5, hwfall_viewer_min_db->y(),
			70, 22, "Filter");
			hwfall_lowpass->add("None|Slow|Med'|Fast");
			hwfall_lowpass->value(progdefaults.hwfall_average_selection);
			hwfall_lowpass->align(Fl_Align(FL_ALIGN_BOTTOM));
			hwfall_lowpass->callback((Fl_Callback*)cb_hwfall_lowpass);

		btn_hwfall_clear_display = new Fl_Button(
			hwfall_lowpass->x() + hwfall_lowpass->w() + 5, hwfall_viewer_min_db->y(),
			50, 22, "Clear");
		btn_hwfall_clear_display->callback((Fl_Callback*)cb_hwfall_clear_display);

		btn_pause_display = new Fl_Light_Button(
			btn_hwfall_clear_display->x() + btn_hwfall_clear_display->w() + 5, hwfall_viewer_min_db->y(),
			55, 22, "Pause");

		hwfall_viewer_palette = new Fl_Choice(
			btn_pause_display->x() + btn_pause_display->w() + 5, hwfall_viewer_min_db->y(),
			80, 22, "Color Palette");
		for (int n = 0; n < 27; n++) \
			hwfall_viewer_palette->add(hwfall::palettes[n].name.c_str());
		hwfall_viewer_palette->align(Fl_Align(FL_ALIGN_BOTTOM));
		hwfall_viewer_palette->value(progdefaults.hwfall_palette);
		hwfall_viewer_palette->callback((Fl_Callback*)cb_hwfall_viewer_palette);

	w->end();

  return w;
}

std::vector<double> hwfall_slice;
PLOT_XY data[NUM_BINS];

static void update_signal_plot(void *)
{
	hwfall_fft_plot->data_1(data, NUM_BINS);
	hwfall_fft_plot->redraw();
}

static void update_wfall(void *)
{
	hwfall_wfall->add_slice(hwfall_slice);
}

void update_frequency()
{
	std::string testmode = qso_opMODE->value();
	int mdoffset = 0;

	if (testmode.find("CW") != std::string::npos)
		mdoffset = progdefaults.CWsweetspot;
	long long freq = wf->rfcarrier();
	if (wf->USB()) freq -= mdoffset;
	else           freq += mdoffset;

	hwfall_wfall->set_fmin( freq );
	hwfall_wfall->set_USB ( wf->USB() );
	hwfall_wfall->set_miny ( hwfall_viewer_start->value() );
}

double maxbin = 0;
double gain = 0.05;
void hwfall_data_update(double *bins, int len)
{
	int p = 0;
	double db;
	if (!hwfall_wfall) return;

	hwfall_slice.assign(NUM_BINS, 0);

	for (int n = 0; n < NUM_BINS; n++) {
		db = (20 * log10(bins[n] + 1e-10) - progdefaults.hwfall_viewer_min_db) / progdefaults.hwfall_viewer_range;
		data[n].x = 100 * db;
		data[n].y = n;

		p = NUM_BINS - 1 - n;
		hwfall_slice[p] = db;
		if (hwfall_slice[p] < 0) hwfall_slice[p] = 0;
		if (hwfall_slice[p] > 1) hwfall_slice[p] = 1;

	}

	update_frequency();
	Fl::awake(update_signal_plot);

	if (disp_time >= disp_next) {
		if (!btn_pause_display->value())
			Fl::awake( update_wfall );
		disp_next += hwfall_waterfall_speed->value();
	}
	disp_time += 1.0 * len / active_modem->get_samplerate();
}

static pthread_mutex_t	hwfall_mutex = PTHREAD_MUTEX_INITIALIZER;

void hwfall_init()
{
	guard_lock sfft_lock( &hwfall_mutex );
	if (!hwfall_wfall) return;

	delete hwfall_viewer_sfft;
	hwfall_start_freq = hwfall_viewer_start->value();
	hwfall_viewer_sfft = new sfft( active_modem->get_samplerate(), hwfall_start_freq, hwfall_start_freq + NUM_BINS);

	update_frequency();
}

int hwfall_viewer_rx_process(const double *buf, int len)
{
	if (!hwfall_wfall || !hwfall_wfall->visible()) return 0;

	double kf = 800.0 / active_modem->get_samplerate();
	cmplx cmplx_in;

	{
		guard_lock sfft_lock( &hwfall_mutex );
		for (int i = 0; i < len; i++) {
			cmplx_in = cmplx(buf[i], buf[i]);
			hwfall_viewer_sfft->run(cmplx_in, sfft_bins, 1);
		}
	}
	if (average_selection != progdefaults.hwfall_average_selection && progdefaults.hwfall_average_selection) {
		average_selection = progdefaults.hwfall_average_selection;
		for (size_t n = 0; n < NUM_BINS; n++) {
			delete hwfall_viewer_movavg[n];
			hwfall_viewer_movavg[n] = new Cmovavg(average_over[average_selection]);
		}
	}
	for (size_t n = 0; n < NUM_BINS; n++) {
		if (progdefaults.hwfall_average_selection)
			bins[n] = kf * hwfall_viewer_movavg[n]->run(norm(sfft_bins[n]));
		else
			bins[n] = kf * norm(sfft_bins[n]);
	}
	hwfall_data_update(bins, len);

	return 0;
}
