// ----------------------------------------------------------------------------
// qrss.cxx  --  VERY LONG DURATION morse code modem
//
// Copyright (C) 2006-2010
//		Dave Freese, W1HKJ
//		   (C) Mauri Niininen, AG1LE
//
// This file is part of fldigi.  Adapted from code contained in gmfsk source code
// distribution.
//  gmfsk Copyright (C) 2001, 2002, 2003
//  Tomi Manninen (oh2bns@sral.fi)
//  Copyright (C) 2004
//  Lawrence Glaister (ve7it@shaw.ca)
//
// Fldigi is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Fldigi is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with fldigi.  If not, see <http://www.gnu.org/licenses/>.
// ----------------------------------------------------------------------------

#define QRSS_DEBUG 0

#include <config.h>

#include <cstring>
#include <string>
#include <stdio.h>
#include <iostream>
#include <fstream>
#include <cstdlib>

#include "timeops.h"
#ifdef __MINGW32__
#  include "compat.h"
#endif

#if !HAVE_CLOCK_GETTIME
#  ifdef __APPLE__
#    include <mach/mach_time.h>
#  endif
#endif

#include "digiscope.h"
#include "waterfall.h"
#include "fl_digi.h"
#include "fftfilt.h"
#include "serial.h"
#include "ptt.h"
#include "main.h"

#include "qrss.h"
#include "misc.h"
#include "configuration.h"
#include "confdialog.h"
#include "status.h"
#include "debug.h"
#include "FTextRXTX.h"
#include "modem.h"

#include "qrunner.h"

#include "winkeyer.h"
#include "nanoIO.h"
#include "KYkeying.h"
#include "ICOMkeying.h"
#include "YAESUkeying.h"

#include "audio_alert.h"

#include "gpio_common.h"

#include "cw.h"

#include "hwfall_viewer.h"

void start_qrssio_thread();
void stop_qrssio_thread();

#if USE_LIBGPIOD

void start_gpio_thread();
void stop_gpio_thread();
static gpio_num_t qrss_gpio_num = GPIO_COMMON_UNKNOWN;

#endif

void qrss::tx_init()
{
	qrss_phase = 0;
	qsk_phase = 0;
	tx_frequency = progdefaults.CWsweetspot;
}

void qrss::rx_init()
{
	frequency = progdefaults.CWsweetspot;
}

void qrss::init()
{

	memset(outbuf, 0, OUTBUFSIZE*sizeof(*outbuf));
	memset(qskbuf, 0, OUTBUFSIZE*sizeof(*qskbuf));

	morse->init();

	rx_init();

}

qrss::~qrss() {
	delete qrss_filter;

	for (size_t n = 0; n < NUM_BINS; n++) delete binfilter[n];
//	delete binfilter;

	stop_qrssio_thread();
#if USE_LIBGPIOD
	stop_gpio_thread();
#endif
}

qrss::qrss() : modem()
{
	qrss_filter = new sfft(SFFT_SIZE, FIRST_BIN, LAST_BIN);
	mode = MODE_QRSS;

	samplerate = QRSS_SAMPLERATE;

	for (size_t n = 0; n < NUM_BINS; n++) binfilter[n] = new Cmovavg(4);

	qrss_speed = 1.2 / progdefaults.qrss_dot;
	qrss_dot_length = 1000.0 * progdefaults.qrss_dot;

	symbollen = (int)round(samplerate * 1.2 / qrss_speed);  // transmit char rate

	qrss_phase = 0.0;
	qsk_phase = 0.0;

	init();

	put_MODEstatus("QRSS");

/*
std::cout <<
"QRSS_SAMPLERATE    : " << QRSS_SAMPLERATE  << std::endl <<
"SFFT_SIZE          : " << SFFT_SIZE  << std::endl <<
"QRSS_F0            : " << QRSS_F0  << std::endl <<
"QRSS_HZPERBIN      : " << QRSS_HZPERBIN  << std::endl <<
"QRSS_BW2           : " << QRSS_BW2  << std::endl <<
"FIRST_BIN          : " << FIRST_BIN  << std::endl <<
"LAST_BIN           : " << LAST_BIN  << std::endl <<
"NUM_BINS           : " << NUM_BINS << std::endl;
*/
}

cmplx qrss::mixer(cmplx in)
{
	static double phase = 0;
	cmplx z (cos(phase), sin(phase));
	z = z * in;

	phase += TWOPI * frequency / samplerate;
	if (phase > TWOPI) phase -= TWOPI;

	return z;
}

int process_count = 0;
int process_times[] = { 2, 4, 8, 16 };
int qrss::rx_process(const double *buf, int len)
{
	return 0;
}

//=====================================================================
// qrss transmit routines
//=====================================================================

//---------------------------------------------------------------------
// qrss_txprocess()
// Read characters from screen and either generate a sound card signal
// at the current transmit audio frequency, or toggle a keyline signa.
// This is called repeatedly from the Rx/Tx thread during tx.
//---------------------------------------------------------------------
int qrss::tx_process()
{
	int c = get_tx_char();

	if (c == GET_TX_CHAR_NODATA) {
//		if (stopflag) {
//			stopflag = false;
//			put_echo_char('\n');
//			return -1;
//		}
		Fl::awake();
		MilliSleep(50);
		return 0;
	}

	if (c == GET_TX_CHAR_ETX) { // || stopflag) {
//		stopflag = false;
		put_echo_char('\n');
		return -1;
	}

	if (CW_KEYLINE_isopen ||
		progdefaults.CW_KEYLINE_on_cat_port ||
		progdefaults.CW_KEYLINE_on_ptt_port ||
		progdefaults.use_FLRIGkeying ) {
		send_via_keyline(c);
		MilliSleep(50);
	} else
		send_tones(c);

#if USE_LIBGPIOD
//	if (progdefaults.gpio_qrss_line != -1)
//		send_gpio_QRSS(c);
#endif

	put_echo_char(c);

	return 0;
}

//----------------------------------------------------------------------
// send morse character using audio tones
//----------------------------------------------------------------------

double qrss::nco(double freq)
{
	qrss_phase += TWOPI * freq / samplerate;

	if (qrss_phase > TWOPI) qrss_phase -= TWOPI;

	return cos(qrss_phase);
}

double qrss::qsknco()
{
	qsk_phase += TWOPI * 1000 / samplerate;

	if (qsk_phase > TWOPI) qsk_phase -= TWOPI;

	return (sin(qsk_phase) > 0 ? 1.0 : -1.0);

}

//---------------------------------------------------------------------
// send_symbol()
// Sends a part of a morse character at the correct freq or silence.
//
// Left channel contains the QRSS waveform
//
// Right channel contains a square wave signal that is used
// to trigger a qsk switch.
//---------------------------------------------------------------------

void qrss::send_symbol(int bit, long len)
{
	double qsk_amp = progdefaults.QSK ? progdefaults.QSKamp : 0.0;
	double xmt_freq = get_txfreq_woffset();

	memset(outbuf, 0, OUTBUFSIZE*sizeof(*outbuf));
	memset(qskbuf, 0, OUTBUFSIZE*sizeof(*qskbuf));

long completed = 0;

	if (bit == 1) { // keydown
		for (int n = 0; n < len; n++) {
			if (n > 0 && n % OUTBUFSIZE == 0) {
				if (progdefaults.QSK)
					ModulateStereo(outbuf, qskbuf, OUTBUFSIZE);
				else
					ModulateXmtr(outbuf, OUTBUFSIZE);
				memset(outbuf, 0, OUTBUFSIZE*sizeof(*outbuf));
				memset(qskbuf, 0, OUTBUFSIZE*sizeof(*qskbuf));
				completed += OUTBUFSIZE;
			}
			outbuf[n % OUTBUFSIZE] = nco(xmt_freq);
			qskbuf[n % OUTBUFSIZE] = qsk_amp * qsknco();
		}
		if (completed < len) {
			if (progdefaults.QSK)
				ModulateStereo(outbuf, qskbuf, len - completed);
			else
				ModulateXmtr(outbuf, len - completed);
		}
	} else { // keyup
		completed = 0;
		memset(outbuf, 0, OUTBUFSIZE*sizeof(*outbuf));
		memset(qskbuf, 0, OUTBUFSIZE*sizeof(*qskbuf));
		while (completed < len) {
			if (progdefaults.QSK)
				ModulateStereo(outbuf, qskbuf, OUTBUFSIZE);
			else
				ModulateXmtr(outbuf, OUTBUFSIZE);
			completed += OUTBUFSIZE;
		}
		completed -= OUTBUFSIZE;
		if (completed < len) {
			if (progdefaults.QSK)
				ModulateStereo(outbuf, qskbuf, len - completed);
			else
				ModulateXmtr(outbuf, len - completed);
		}
	}
}

//---------------------------------------------------------------------
// send_ch()
// sends a morse character and the space afterwards
//---------------------------------------------------------------------
void qrss::send_tones(int ch)
{
	std::string code;

	float stdwpm = 1.2 / progdefaults.qrss_dot;

	float kfactor = QRSS_SAMPLERATE / 1000.0;
	float tc = 1200.0 / stdwpm;
	float tch = 3 * tc, twd = 4 * tc;

	tc *= kfactor;
	tch *= kfactor;
	twd *= kfactor;

	if ((ch == ' ') || (ch == '\n')) {
		while (twd > 0) {
			send_symbol(0, twd < 4096 ? twd : 4096);
			twd -= 4096;
		}
		put_echo_char(progdefaults.rx_lowercase ? tolower(ch) : ch);
		return;
	}

	code = morse->tx_lookup(ch);

	if (!code.length()) {
		return;
	}

	int elements = code.length();

	for (int n = 0; n < elements; n++) {
		send_symbol(1, (code[n] == '-' ? 3 : 1 * symbollen));
		send_symbol(0, ((n < elements - 1) ? tc : tch));
	}

	if (ch != -1) {
		std::string prtstr = morse->tx_print();
		for (size_t n = 0; n < prtstr.length(); n++)
			put_echo_char(
				prtstr[n],
				prtstr[0] == '<' ? FTextBase::CTRL : FTextBase::XMIT);
	}
}

// ---------------------------------------------------------------------
// QRSS output on DTR/RTS signal lines
// or xmlrpc interface to flrig as the DTR/RTS line controller
//----------------------------------------------------------------------

void QRSS_flrig_key(int down)
{
	flrig_key_state(down);
}

bool QRSS_KEYLINE_is_open = false;

void open_qrss_KEYLINE()
{
	if (open_CW_KEYLINE())
		QRSS_KEYLINE_is_open = true;;
}

void close_QRSS_KEYLINE()
{
	close_CW_KEYLINE();
	QRSS_KEYLINE_is_open = false;
}

//----------------------------------------------------------------------
#include <queue>

static pthread_t       qrssio_pthread;
static pthread_cond_t  qrssio_cond;
static pthread_mutex_t qrssio_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t fifo_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t qrssio_ptt_mutex = PTHREAD_MUTEX_INITIALIZER;

static bool qrssio_thread_running   = false;
static bool qrssio_terminate_flag   = false;

//----------------------------------------------------------------------

static int qrssio_ch;
static cMorse *qrssio_morse = 0;
static std::queue<int> fifo;

//----------------------------------------------------------------------

void qrssio_key(int on)
{
	if (progdefaults.use_FLRIGkeying) {
		QRSS_flrig_key(on);
		return;
	}

	if (QRSS_KEYLINE_is_open ||
		progdefaults.CW_KEYLINE_on_cat_port ||
		progdefaults.CW_KEYLINE_on_ptt_port) {

		Cserial *ser = &CW_KEYLINE_serial;

		if (progdefaults.CW_KEYLINE_on_cat_port)
			ser = &rigio;
		else if (progdefaults.CW_KEYLINE_on_ptt_port)
			ser = &push2talk->serPort;
		switch (progdefaults.CW_KEYLINE) {
			case 0: break;
			case 1: ser->SetRTS(on); break;
			case 2: ser->SetDTR(on); break;
		}
	}
}

void qrssio_ptt(int on)
{
	if (progdefaults.use_FLRIGkeying)
		return;

	if (QRSS_KEYLINE_is_open ||
		progdefaults.CW_KEYLINE_on_cat_port ||
		progdefaults.CW_KEYLINE_on_ptt_port) {
		Cserial *ser = &CW_KEYLINE_serial;
		if (progdefaults.CW_KEYLINE_on_cat_port)
			ser = &rigio;
		else if (progdefaults.CW_KEYLINE_on_ptt_port)
			ser = &push2talk->serPort;
		switch (progdefaults.PTT_KEYLINE) {
			case 0: break;
			case 1: ser->SetRTS(on); break;
			case 2: ser->SetDTR(on); break;
		}
	}
}

// return accurate time of day in secs

double qrssio_now()
{
	static struct timespec tp;

#if HAVE_CLOCK_GETTIME
	clock_gettime(CLOCK_MONOTONIC, &tp); 
#elif defined(__WIN32__)
	DWORD msec = GetTickCount();
	return 1.0 * msec;
	tp.tv_sec = msec / 1000;
	tp.tv_nsec = (msec % 1000) * 1000000;
#elif defined(__APPLE__)
	static mach_timebase_info_data_t info = { 0, 0 };
	if (unlikely(info.denom == 0))
		mach_timebase_info(&info);
	uint64_t t = mach_absolute_time() * info.numer / info.denom;
	tp.tv_sec = t / 1000000000;
	tp.tv_nsec = t % 1000000000;
#endif
	return 1.0 * tp.tv_sec + tp.tv_nsec * 1e-9;

}

void qrssio_bit(int bit, double msecs)
{
//std::cout << bit << " : " << msecs << std::endl;
#ifdef QRSS_TTEST
	if (!qrssio_test) qrssio_test = fopen("qrssio_test.txt", "a");
#endif
	static double secs;
	static struct timespec tv = { 0, 1000000L};
	static double end1 = 0;
	static double end2 = 0;
	static double t1 = 0;
#ifdef QRSS_TTEST
	static double t2 = 0;
#endif
	static double t3 = 0;
	static double t4 = 0;
	int loop1 = 0;
	int loop2 = 0;
	int n1 = msecs * 1e3;

	secs = msecs * 1e-3;

#ifdef __WIN32__
	timeBeginPeriod(1);
#endif

	t1 = qrssio_now();

	end2 = t1 + secs - 0.00001;

	qrssio_key(bit);

#ifdef QRSS_TTEST
	t2 = t3 = qrssio_now();
#else
	t3 = qrssio_now();
#endif
	end1 = end2 - 0.005;

	while (t3 < end1 && (++loop1 < n1)) {
		nano_sleep(&tv, NULL);
		t3 = qrssio_now();
	}

	t4 = t3;
	while (t4 <= end2) {
		loop2++;
		t4 = qrssio_now();
	}

#ifdef __WIN32__
	timeEndPeriod(1);
#endif

#ifdef QRSS_TTEST
	if (qrssio_test)
		fprintf(qrssio_test, "%d, %d, %d, %6f, %6f, %6f, %6f, %6f, %6f, %6f\n",
			bit, loop1, loop2,
			secs * 1e3,
			(t2 - t1)*1e3,
			(t3 - t1)*1e3,
			(t3 - end1) * 1e3,
			(t4 - t1)*1e3,
			(t4 - end2) * 1e3,
			(t4 - t1 - secs)*1e3);
#endif
}

bool qrssio_sending = false;
void send_qrssio(int c)
{
	if (c == GET_TX_CHAR_NODATA || c == 0x0d) {
		qrssio_sending = false;
		return;
	}

	qrssio_sending = true;

	float tc = 1000.0 * progdefaults.qrss_dot;
	if (tc <= 0) tc = 1;
	float tch = 3 * tc, twd = 4 * tc;

	if (c == 0x0a) c = ' ';

	if (c == ' ') {
		qrssio_bit(0, twd);
		qrssio_sending = false;
		return;
	}

	std::string code;
	code = qrssio_morse->tx_lookup(c);
	if (!code.length()) {
		qrssio_sending = false;
		return;
	}

	guard_lock lk(&qrssio_ptt_mutex);

	for (size_t n = 0; n < code.length(); n++) {
		if (code[n] == '.') {
			qrssio_bit(1, tc);
		} else {
			qrssio_bit(1, 3*tc);
		}
		if (n < code.length() -1) {
			qrssio_bit(0, tc);
		} else {
			qrssio_bit(0, tch);
		}
	}

	qrssio_sending = false;
}

void * qrssio_loop(void *args)
{
//	SET_THREAD_ID(QRSSIO_TID);

	qrssio_thread_running   = true;
	qrssio_terminate_flag   = false;

	while(1) {
		pthread_mutex_lock(&qrssio_mutex);
		pthread_cond_wait(&qrssio_cond, &qrssio_mutex);
		pthread_mutex_unlock(&qrssio_mutex);

		if (qrssio_terminate_flag)
			break;
		while (!fifo.empty()) {
			{
				guard_lock lk(&fifo_mutex);
				qrssio_ch = fifo.front();
				fifo.pop();
			}
			send_qrssio(qrssio_ch);
		}
	}
	return (void *)0;
}

void stop_qrssio_thread(void)
{
	if(!qrssio_thread_running) return;

	qrssio_terminate_flag = true;
	pthread_cond_signal(&qrssio_cond);

	MilliSleep(10);

	pthread_join(qrssio_pthread, NULL);

	pthread_mutex_destroy(&qrssio_mutex);
	pthread_cond_destroy(&qrssio_cond);

	memset((void *) &qrssio_pthread, 0, sizeof(qrssio_pthread));
	memset((void *) &qrssio_mutex,   0, sizeof(qrssio_mutex));

	qrssio_thread_running   = false;
	qrssio_terminate_flag   = false;

	delete qrssio_morse;
	qrssio_morse = 0;
}

void start_qrssio_thread(void)
{
	if (qrssio_thread_running) return;

	memset((void *) &qrssio_pthread, 0, sizeof(qrssio_pthread));
	memset((void *) &qrssio_mutex,   0, sizeof(qrssio_mutex));
	memset((void *) &qrssio_cond,    0, sizeof(qrssio_cond));

	if(pthread_cond_init(&qrssio_cond, NULL)) {
		LOG_ERROR("Alert thread create fail (pthread_cond_init)");
		return;
	}

	if(pthread_mutex_init(&qrssio_mutex, NULL)) {
		LOG_ERROR("AUDIO_ALERT thread create fail (pthread_mutex_init)");
		return;
	}

	if (pthread_create(&qrssio_pthread, NULL, qrssio_loop, NULL) < 0) {
		pthread_mutex_destroy(&qrssio_mutex);
		LOG_ERROR("AUDIO_ALERT thread create fail (pthread_create)");
	}

	LOG_DEBUG("started audio qrssio thread");

	MilliSleep(10); // Give the CPU time to set 'qrssio_thread_running'
}

void qrss_wait(int c)
{
//	int len = qrssio_morse->tx_length(c);
	while (qrssio_sending) {
		MilliSleep(50);
		Fl::awake();
	}
}

void send_via_keyline(int c)
{
	if (!qrssio_thread_running)
		start_qrssio_thread();

	if (qrssio_morse == 0) {
		qrssio_morse = new cMorse;
		qrssio_morse->init();
	}

	guard_lock lk(&fifo_mutex);
	fifo.push(c);

	pthread_cond_signal(&qrssio_cond);

	qrss_wait(c); // wait for send to complete
}

//----------------------------------------------------------------------
// QRSS output on GPIO pin
//----------------------------------------------------------------------

#if USE_LIBGPIOD

#include <queue>
#include <fcntl.h>

static pthread_t		QRSS_gpio_pthread;
static pthread_cond_t	QRSS_gpio_cond;
static pthread_mutex_t	QRSS_gpio_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t	GPIO_fifo_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t	QRSS_gpio_ptt_mutex = PTHREAD_MUTEX_INITIALIZER;

static bool				QRSS_gpio_thread_running   = false;
static bool				QRSS_gpio_terminate_flag   = false;

static cMorse			*gpio_morse = 0;

static void set_gpio_pin(bool key)
{
	int ret;

	ret = gpio_common_set(qrss_gpio_num, key);

	if (ret < 0) {
		LOG_ERROR("Error setting GPIO");
	}

}

//----------------------------------------------------------------------

static double QRSS_gpio_now()
{
	static struct timespec tp;

#if HAVE_CLOCK_GETTIME
	clock_gettime(CLOCK_MONOTONIC, &tp); 
#elif defined(__WIN32__)
	DWORD msec = GetTickCount();
	return 1.0 * msec;
	tp.tv_sec = msec / 1000;
	tp.tv_nsec = (msec % 1000) * 1000000;
#elif defined(__APPLE__)
	static mach_timebase_info_data_t info = { 0, 0 };
	if (unlikely(info.denom == 0))
		mach_timebase_info(&info);
	uint64_t t = mach_absolute_time() * info.numer / info.denom;
	tp.tv_sec = t / 1000000000;
	tp.tv_nsec = t % 1000000000;
#endif
	return 1.0 * tp.tv_sec + tp.tv_nsec * 1e-9;

}

static void QRSS_gpio_bit(int bit, double msecs)
{
	static double secs;
	static struct timespec tv = { 0, 1000000L};
	static double end1 = 0;
	static double end2 = 0;
	static double t1 = 0;
#ifdef QRSS_gpio_TTEST
	static double t2 = 0;
#endif
	static double t3 = 0;
	static double t4 = 0;
	int loop1 = 0;
	int loop2 = 0;
	int n1 = msecs * 1e3;

	secs = msecs * 1e-3;

#ifdef __WIN32__
	timeBeginPeriod(1);
#endif

	t1 = QRSS_gpio_now();

	end2 = t1 + secs - 0.00001;

	set_gpio_pin(bit);

#ifdef QRSS_gpio_TTEST
	t2 = t3 = QRSS_gpio_now();
#else
	t3 = QRSS_gpio_now();
#endif
	end1 = end2 - 0.005;

	while (t3 < end1 && (++loop1 < n1)) {
		nano_sleep(&tv, NULL);
		t3 = QRSS_gpio_now();
	}

	t4 = t3;
	while (t4 <= end2) {
		loop2++;
		t4 = QRSS_gpio_now();
	}

#ifdef __WIN32__
	timeEndPeriod(1);
#endif

}

static void send_gpio(int c)
{
	if (c == GET_TX_CHAR_NODATA || c == 0x0d) {
		return;
	}

	float qrss_speed;
	if (progdefaults.QRSS_qrss)
		qrss_speed = 1.2 / progdefaults.QRSS_qrss_dot;
	else
		qrss_speed  = progdefaults.QRSSspeed;

	float tc = 1200.0 / qrss_speed;
	if (tc <= 0) tc = 1;
	float ta = 0.0;
	float tch = 3 * tc, twd = 4 * tc;
	float   stdwpm = qrss_speed;

	if (!progdefaults.QRSS_qrss) {
		if (progdefaults.QRSSusefarnsworth && (stdwpm > progdefaults.QRSSfarnsworth)) {
			ta = 60000.0 / progdefaults.QRSSfarnsworth - 37200.0 / progdefaults.QRSSspeed;
			tch = 3 * ta / 19;
			twd = 4 * ta / 19;
			stdwpm = progdefaults.QRSSfarnsworth;
		}
		if (progdefaults.QRSSusewordsworth && (stdwpm > progdefaults.QRSSwordsworth)) {
			twd += (50.0 * 1200.0 / progdefaults.QRSSwordsworth - 50.0 * 1200.0 / stdwpm);
		}
	}

	if (c == 0x0a) c = ' ';

	if (c == ' ') {
		QRSS_gpio_bit(0, twd);
		return;
	}

	std::string code;
	code = gpio_morse->tx_lookup(c);
	if (!code.length()) {
		return;
	}

	guard_lock lk(&QRSS_gpio_ptt_mutex);

	double weight = progdefaults.QRSSweight;

	for (size_t n = 0; n < code.length(); n++) {
		if (code[n] == '.') {
			QRSS_gpio_bit(1, tc * (1 + weight));
		} else {
			QRSS_gpio_bit(1, 3*tc * (1 + weight));
		}
		if (n < code.length() -1) {
			QRSS_gpio_bit(0, tc * (1 - weight));
		} else {
			QRSS_gpio_bit(0, tch - tc * weight);
		}
	}
}

static void * QRSS_gpio_loop(void *args)
{
//	SET_THREAD_ID(QRSS_gpio_TID);

	QRSS_gpio_thread_running   = true;
	QRSS_gpio_terminate_flag   = false;
	int QRSS_gpio_ch;

	while(1) {
		pthread_mutex_lock(&QRSS_gpio_mutex);
		pthread_cond_wait(&QRSS_gpio_cond, &QRSS_gpio_mutex);
		pthread_mutex_unlock(&QRSS_gpio_mutex);

		if (QRSS_gpio_terminate_flag)
			break;
		while (!fifo.empty()) {
			{
				guard_lock lk(&GPIO_fifo_mutex);
				QRSS_gpio_ch = fifo.front();
				fifo.pop();
			}
			send_gpio(QRSS_gpio_ch);
		}
	}
	return (void *)0;
}

void stop_gpio_thread(void)
{
	if(!QRSS_gpio_thread_running) return;

	QRSS_gpio_terminate_flag = true;
	pthread_cond_signal(&QRSS_gpio_cond);

	MilliSleep(10);

	pthread_join(QRSS_gpio_pthread, NULL);

	pthread_mutex_destroy(&QRSS_gpio_mutex);
	pthread_cond_destroy(&QRSS_gpio_cond);

	memset((void *) &QRSS_gpio_pthread, 0, sizeof(QRSS_gpio_pthread));
	memset((void *) &QRSS_gpio_mutex,   0, sizeof(QRSS_gpio_mutex));

	delete gpio_morse;

	QRSS_gpio_thread_running   = false;
	QRSS_gpio_terminate_flag   = false;

	if (qrss_gpio_num != GPIO_COMMON_UNKNOWN)
		gpio_common_close(qrss_gpio_num);
	qrss_gpio_num = GPIO_COMMON_UNKNOWN;

}

void start_gpio_thread(void)
{
	if (QRSS_gpio_thread_running) return;

	memset((void *) &QRSS_gpio_pthread, 0, sizeof(QRSS_gpio_pthread));
	memset((void *) &QRSS_gpio_mutex,   0, sizeof(QRSS_gpio_mutex));
	memset((void *) &QRSS_gpio_cond,    0, sizeof(QRSS_gpio_cond));

	if(pthread_cond_init(&QRSS_gpio_cond, NULL)) {
		LOG_ERROR("GPIO thread create fail (pthread_cond_init)");
		return;
	}

	if(pthread_mutex_init(&QRSS_gpio_mutex, NULL)) {
		LOG_ERROR("GPIO thread create fail (pthread_mutex_init)");
		return;
	}

	if (pthread_create(&QRSS_gpio_pthread, NULL, QRSS_gpio_loop, NULL) < 0) {
		pthread_mutex_destroy(&QRSS_gpio_mutex);
		LOG_ERROR("GPIO thread create fail (pthread_create)");
	}

	int time_out = 100;
	while (!QRSS_gpio_thread_running) {
		MilliSleep(10);
		if (--time_out == 0) {
			LOG_ERROR("Could not start QRSS GPIO thread");
			return;
		}
	}

	if (gpio_morse == 0) {
		gpio_morse = new cMorse;
		gpio_morse->init();
	}

	LOG_INFO("Started QRSS GPIO thread");

}

void qrss::send_gpio_QRSS(int c)
{
	if (!progdefaults.enable_gpio_qrss || (progdefaults.gpio_qrss_line == -1)) return;

	if (!QRSS_gpio_thread_running) {
		LOG_INFO("Opening GPIO for QRSS keying");
		qrss_gpio_num = gpio_common_open_line(progdefaults.gpio_qrss_device.c_str(), progdefaults.gpio_qrss_line, false);
		start_gpio_thread();
	}
	guard_lock lk(&GPIO_fifo_mutex);
	fifo.push(c);

	pthread_cond_signal(&QRSS_gpio_cond);

}

#endif
